// Configuração do Firebase para notificações push
// Substitua os valores abaixo pelos valores do seu projeto Firebase

const firebaseConfig = {
  apiKey: "SUA_API_KEY",
  authDomain: "seu-projeto.firebaseapp.com",
  projectId: "seu-projeto",
  storageBucket: "seu-projeto.appspot.com",
  messagingSenderId: "SEU_MESSAGING_SENDER_ID",
  appId: "SEU_APP_ID"
};

// Inicializar o Firebase
if (typeof firebase !== 'undefined') {
  firebase.initializeApp(firebaseConfig);
} else {
  console.error('Firebase SDK não carregado. Verifique se você incluiu o script do Firebase.');
}

// Função para solicitar permissão de notificação e obter token FCM
async function requestNotificationPermission() {
  try {
    // Verificar se o Firebase Messaging está disponível
    if (typeof firebase === 'undefined' || !firebase.messaging) {
      console.error('Firebase Messaging não está disponível.');
      return null;
    }
    
    const messaging = firebase.messaging();
    
    // Solicitar permissão
    await Notification.requestPermission();
    
    // Obter token
    const token = await messaging.getToken({
      vapidKey: 'SUA_CHAVE_VAPID_PUBLICA'
    });
    
    if (token) {
      console.log('Token FCM obtido:', token);
      return token;
    } else {
      console.error('Não foi possível obter o token.');
      return null;
    }
  } catch (error) {
    console.error('Erro ao solicitar permissão ou obter token:', error);
    return null;
  }
}

// Função para registrar o token no servidor
async function registerTokenWithServer(token) {
  try {
    const response = await fetch('index.php?action=register', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ token }),
    });
    
    const data = await response.json();
    
    if (data.status === 'success') {
      console.log('Token registrado com sucesso no servidor');
      return true;
    } else {
      console.error('Erro ao registrar token no servidor:', data.error);
      return false;
    }
  } catch (error) {
    console.error('Erro ao enviar token para o servidor:', error);
    return false;
  }
}

// Configurar o listener para mensagens em primeiro plano
function setupForegroundNotificationListener() {
  if (typeof firebase === 'undefined' || !firebase.messaging) {
    console.error('Firebase Messaging não está disponível.');
    return;
  }
  
  const messaging = firebase.messaging();
  
  // Lidar com mensagens quando o aplicativo está em primeiro plano
  messaging.onMessage((payload) => {
    console.log('Mensagem recebida em primeiro plano:', payload);
    
    // Exibir notificação usando o Service Worker
    if ('serviceWorker' in navigator && navigator.serviceWorker.controller) {
      navigator.serviceWorker.controller.postMessage({
        type: 'SHOW_NOTIFICATION',
        title: payload.notification?.title || 'Nova Notificação',
        message: payload.notification?.body || 'Você recebeu uma nova notificação.',
        data: payload.data || {}
      });
    }
  });
}

// Exportar funções para uso global
window.firebaseNotifications = {
  requestPermission: requestNotificationPermission,
  registerToken: registerTokenWithServer,
  setupForegroundListener: setupForegroundNotificationListener
};