#!/bin/bash
# Script para gerar ícones para o PWA a partir de uma única imagem
# Requer o ImageMagick instalado: sudo apt-get install imagemagick

# Cores para saída
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Função para exibir mensagens
print_message() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[AVISO]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERRO]${NC} $1"
}

# Verificar se o ImageMagick está instalado
if ! command -v convert &> /dev/null; then
    print_error "ImageMagick não está instalado. Por favor, instale-o primeiro:"
    print_error "sudo apt-get install imagemagick"
    exit 1
fi

# Verificar se um arquivo de imagem foi fornecido
if [ $# -eq 0 ]; then
    print_error "Nenhum arquivo de imagem fornecido."
    print_message "Uso: $0 caminho/para/imagem.png"
    exit 1
fi

# Verificar se o arquivo existe
if [ ! -f "$1" ]; then
    print_error "O arquivo '$1' não existe."
    exit 1
fi

# Criar diretório de ícones se não existir
if [ ! -d "icons" ]; then
    print_message "Criando diretório 'icons'..."
    mkdir -p icons
fi

# Tamanhos de ícones necessários para o PWA
SIZES=(72 96 128 144 152 192 384 512)

# Gerar ícones em todos os tamanhos
for size in "${SIZES[@]}"; do
    output_file="icons/icon-${size}x${size}.png"
    print_message "Gerando ícone ${size}x${size}..."
    
    convert "$1" -resize ${size}x${size} "$output_file"
    
    if [ $? -ne 0 ]; then
        print_error "Falha ao gerar o ícone ${size}x${size}."
    else
        print_message "Ícone ${size}x${size} gerado com sucesso: $output_file"
    fi
done

# Gerar favicon.ico
print_message "Gerando favicon.ico..."
convert "$1" -resize 16x16 -resize 32x32 -resize 48x48 -colors 256 "icons/favicon.ico"

if [ $? -ne 0 ]; then
    print_error "Falha ao gerar o favicon.ico."
else
    print_message "favicon.ico gerado com sucesso: icons/favicon.ico"
fi

print_message "Todos os ícones foram gerados com sucesso!"
print_message "Os ícones estão disponíveis no diretório 'icons/'."