<?php
// Configurações
$config = [
    'debug' => true,
    'log_file' => 'webhook_log.txt',
    'firebase_server_key' => 'SUA_CHAVE_FIREBASE_AQUI', // Substitua pela sua chave real
    'allowed_origins' => ['*'] // Em produção, restrinja para origens específicas
];

// Habilitar log de erros em modo debug
if ($config['debug']) {
    ini_set('display_errors', 1);
    error_reporting(E_ALL);
}

// Função para registrar logs
function logMessage($message) {
    global $config;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] $message\n";
    file_put_contents($config['log_file'], $logEntry, FILE_APPEND);
}

// Função para enviar resposta JSON
function sendJsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

// Verificar se é uma requisição POST (webhook)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Obter o corpo da requisição
    $requestBody = file_get_contents('php://input');
    $webhookData = json_decode($requestBody, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        logMessage("Erro ao decodificar JSON: " . json_last_error_msg());
        sendJsonResponse(['error' => 'JSON inválido'], 400);
    }
    
    // Registrar o webhook recebido
    logMessage("Webhook recebido: " . $requestBody);
    
    // Processar os dados do webhook
    $title = isset($webhookData['title']) ? $webhookData['title'] : 'Nova notificação';
    $message = isset($webhookData['message']) ? $webhookData['message'] : 'Você recebeu uma nova notificação.';
    $data = isset($webhookData['data']) ? $webhookData['data'] : [];
    
    // Buscar tokens de dispositivo do banco de dados (simplificado para este exemplo)
    // Em uma implementação real, você buscaria os tokens do banco de dados
    $deviceTokens = [];
    if (file_exists('device_tokens.json')) {
        $deviceTokens = json_decode(file_get_contents('device_tokens.json'), true);
    }
    
    if (empty($deviceTokens)) {
        logMessage("Nenhum dispositivo registrado para receber notificações.");
        sendJsonResponse(['status' => 'warning', 'message' => 'Nenhum dispositivo registrado']);
    }
    
    // Enviar notificações para cada dispositivo
    $successCount = 0;
    $errorCount = 0;
    
    foreach ($deviceTokens as $token) {
        $result = sendPushNotification($token, $title, $message, $data, $config);
        if ($result['success']) {
            $successCount++;
        } else {
            $errorCount++;
            logMessage("Erro ao enviar notificação: " . $result['error']);
        }
    }
    
    sendJsonResponse([
        'status' => 'success',
        'message' => "Notificações processadas: $successCount enviadas, $errorCount falhas"
    ]);
} 
// Rota para registrar um novo dispositivo
elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['action']) && $_GET['action'] === 'register') {
    $requestBody = file_get_contents('php://input');
    $data = json_decode($requestBody, true);
    
    if (!isset($data['token']) || empty($data['token'])) {
        sendJsonResponse(['error' => 'Token não fornecido'], 400);
    }
    
    $token = $data['token'];
    $deviceTokens = [];
    
    if (file_exists('device_tokens.json')) {
        $deviceTokens = json_decode(file_get_contents('device_tokens.json'), true);
    }
    
    if (!in_array($token, $deviceTokens)) {
        $deviceTokens[] = $token;
        file_put_contents('device_tokens.json', json_encode($deviceTokens));
        logMessage("Novo dispositivo registrado: $token");
    }
    
    sendJsonResponse(['status' => 'success', 'message' => 'Dispositivo registrado com sucesso']);
}
// Servir a página principal
else {
    // Redirecionar para a página principal
    include 'app.html';
}

/**
 * Função para enviar notificação push usando Firebase Cloud Messaging
 */
function sendPushNotification($token, $title, $message, $additionalData = [], $config) {
    $url = 'https://fcm.googleapis.com/fcm/send';
    
    $notification = [
        'title' => $title,
        'body' => $message,
        'icon' => 'notification-icon.png',
        'click_action' => 'OPEN_APP'
    ];
    
    $fcmNotification = [
        'to' => $token,
        'notification' => $notification,
        'data' => $additionalData
    ];
    
    $headers = [
        'Authorization: key=' . $config['firebase_server_key'],
        'Content-Type: application/json'
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($fcmNotification));
    
    $result = curl_exec($ch);
    
    if ($result === false) {
        return ['success' => false, 'error' => curl_error($ch)];
    }
    
    curl_close($ch);
    
    $response = json_decode($result, true);
    
    if (isset($response['success']) && $response['success'] == 1) {
        return ['success' => true];
    } else {
        return ['success' => false, 'error' => isset($response['results'][0]['error']) ? $response['results'][0]['error'] : 'Erro desconhecido'];
    }
}