#!/bin/bash
# Script de instalação para o N8N Notificador
# Este script configura um servidor web Apache com PHP em um servidor Ubuntu/Debian

# Cores para saída
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Função para exibir mensagens
print_message() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[AVISO]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERRO]${NC} $1"
}

# Verificar se está sendo executado como root
if [ "$EUID" -ne 0 ]; then
    print_error "Este script precisa ser executado como root (sudo)."
    exit 1
fi

# Atualizar repositórios
print_message "Atualizando repositórios..."
apt-get update

# Instalar Apache, PHP e extensões necessárias
print_message "Instalando Apache, PHP e extensões necessárias..."
apt-get install -y apache2 php libapache2-mod-php php-json php-curl php-mbstring php-xml

# Verificar se a instalação foi bem-sucedida
if [ $? -ne 0 ]; then
    print_error "Falha ao instalar os pacotes necessários."
    exit 1
fi

# Habilitar módulos do Apache necessários
print_message "Habilitando módulos do Apache..."
a2enmod rewrite headers ssl

# Reiniciar o Apache para aplicar as alterações
print_message "Reiniciando o Apache..."
systemctl restart apache2

# Verificar se o Apache está em execução
if ! systemctl is-active --quiet apache2; then
    print_error "O Apache não está em execução. Verifique os logs para mais detalhes."
    exit 1
fi

# Criar diretório para o aplicativo
print_message "Configurando diretório do aplicativo..."
APP_DIR="/var/www/html/n8n-notifier"

# Verificar se o diretório já existe
if [ -d "$APP_DIR" ]; then
    print_warning "O diretório $APP_DIR já existe. O conteúdo será substituído."
    rm -rf "$APP_DIR"
fi

# Criar o diretório
mkdir -p "$APP_DIR"
mkdir -p "$APP_DIR/icons"

# Copiar arquivos do aplicativo
print_message "Copiando arquivos do aplicativo..."
cp -r ./* "$APP_DIR/"

# Definir permissões corretas
print_message "Configurando permissões..."
chown -R www-data:www-data "$APP_DIR"
chmod -R 755 "$APP_DIR"
chmod -R 777 "$APP_DIR" # Temporariamente para permitir a criação de arquivos de log e tokens

# Criar arquivo de configuração do site
print_message "Criando configuração do site para o Apache..."
cat > /etc/apache2/sites-available/n8n-notifier.conf << EOF
<VirtualHost *:80>
    ServerAdmin webmaster@localhost
    DocumentRoot $APP_DIR
    ServerName n8n-notifier.seu-dominio.com
    
    <Directory $APP_DIR>
        Options Indexes FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>
    
    ErrorLog \${APACHE_LOG_DIR}/n8n-notifier-error.log
    CustomLog \${APACHE_LOG_DIR}/n8n-notifier-access.log combined
</VirtualHost>
EOF

# Habilitar o site
print_message "Habilitando o site..."
a2ensite n8n-notifier.conf

# Reiniciar o Apache para aplicar as alterações
print_message "Reiniciando o Apache..."
systemctl restart apache2

# Verificar se o PHP está funcionando corretamente
print_message "Verificando a instalação do PHP..."
PHP_VERSION=$(php -v | grep -oP '(?<=PHP )[0-9]+\.[0-9]+\.[0-9]+' | head -1)

if [ -z "$PHP_VERSION" ]; then
    print_error "Não foi possível determinar a versão do PHP. Verifique se o PHP está instalado corretamente."
else
    print_message "PHP $PHP_VERSION instalado com sucesso."
fi

# Instruções finais
print_message "Instalação concluída!"
print_message "Seu aplicativo está disponível em: http://seu-servidor/n8n-notifier/"
print_message ""
print_message "Próximos passos:"
print_message "1. Edite o arquivo $APP_DIR/index.php e atualize a chave do Firebase"
print_message "2. Edite o arquivo $APP_DIR/firebase-config.js com suas configurações do Firebase"
print_message "3. Adicione os ícones na pasta $APP_DIR/icons"
print_message "4. Configure um certificado SSL (recomendado para PWA e notificações push)"
print_message "5. Atualize o arquivo de configuração do Apache em /etc/apache2/sites-available/n8n-notifier.conf"
print_message "   com seu nome de domínio correto"
print_message ""
print_warning "IMPORTANTE: Para produção, você deve configurar HTTPS, pois as notificações push"
print_warning "e PWAs requerem uma conexão segura. Você pode usar Let's Encrypt para obter"
print_warning "um certificado SSL gratuito."
print_message ""
print_message "Para instalar o Let's Encrypt, execute:"
print_message "apt-get install certbot python3-certbot-apache"
print_message "certbot --apache -d n8n-notifier.seu-dominio.com"
print_message ""
print_message "Após configurar tudo, restrinja as permissões do diretório:"
print_message "chmod -R 755 $APP_DIR"
print_message "chmod 777 $APP_DIR/device_tokens.json $APP_DIR/webhook_log.txt"